/// <reference path="/njres/js/comlib/neo.js.src/000neo.js" />

/// <summary>
///   日時処理
///
///   日付フォーマットの参考URL
///   http://unicode.org/reports/tr35/tr35-6.html#Date_Format_Patterns
///   http://java.sun.com/javase/ja/6/docs/ja/api/java/text/SimpleDateFormat.html
/// </summary>
neo.dateTime = {
    parse: function (dateString) {
        /// <summary>文字列から日付を取得する。正しくない場合はnull</summary>
        /// <param name="dateString" type="String">日付文字列</param>
        /// <returns type="Date" />
        dateString = String(dateString);

        for (var i = 0; i < this._parsers.length; i++) {
            var date = this._parsers[i].apply(this, [dateString]);
            if (date != null) {
                return date;
            }
        }

        return null;
    },

    format: function (format, date) {
        /// <summary>日付を整形した文字列を取得する</summary>
        /// <param name="format" type="String">整形文字列(記号:yMdEahHms)</param>
        /// <param name="date" type="Date">日付、省略すると今日</param>
        /// <returns type="String" />

        var result = "";
        var inLiteral = false;
        var formatter = $.extend({}, this._formatter, {
            "date": !!date ? date : new Date()
        });
        // 1文字ずつ処理
        for (var i = 0; i < format.length; i++) {
            var curr = format.charAt(i);
            if (curr == "'") {
                // 文字列部の切れ目
                inLiteral = !inLiteral;
                result += inLiteral ? formatter.beginLiteral() : formatter.endLiteral();
            } else if (inLiteral) {
                // 文字列部内部
                formatter.leteral += curr;
            } else if (!formatter.isSymbol(curr)) {
                // 記号以外の文字
                result += formatter.getCurrentToken() + curr;
            } else if (curr != formatter.symbolChar) {
                // 新しい記号部の先頭
                result += formatter.beginSymbol(curr);
            } else {
                // 記号部の途中
                formatter.symbolLength++;
            }
        }
        // 最後のトークンの処理
        result += formatter.getCurrentToken();
        return result;
    },

    compareOn: function (date1, date2, compName) {
        /// <summary>日付を比較する(-1=date1の方が小さい、0=等しい、1=date1の方が大きい)</summary>
        /// <param name="date1" type="Date">日付1</param>
        /// <param name="date2" type="Date">日付2</param>
        /// <param name="compName" type="String">比較対象(year, month, day, hour, minute, second=既定)</param>
        /// <returns type="Number" />
        var comps = {
            'year': function (d) { return d.getFullYear(); },
            'month': function (d) { return d.getMonth(); },
            'day': function (d) { return d.getDate(); },
            'hour': function (d) { return d.getHours(); },
            'minute': function (d) { return d.getMinutes(); },
            'second': function (d) { return d.getSeconds(); }
        };

        for (var key in comps) {
            var result = comps[key](date1) - comps[key](date2);
            if (result != 0) {
                return result / Math.abs(result);
            }
            if (compName == key) {
                break;
            }
        }

        return 0;
    },

    moveDate: function (date, delta) {
        /// <summary>基準日付のN日後、N日前を取得する</summary>
        /// <param name="date" type="Date">基準日付</param>
        /// <param name="delta" type="Number">加算日</param>
        /// <returns type="Date" />
        var movedDate = new Date(date.getTime());
        movedDate.setDate(date.getDate() + delta);
        return movedDate;
    },

    moveMonth: function (date, delta) {
        /// <summary>基準日付のNヶ月後、Nヶ月前を取得する</summary>
        /// <param name="date" type="Date">基準日付</param>
        /// <param name="delta" type="Number">加算月</param>
        /// <returns type="Date" />

        var ym = date.getFullYear() * 12 + date.getMonth() + delta;
        var year = Math.floor(ym / 12);
        var month = ym % 12;
        var day = date.getDate();
        var endDay = this.getMonthEndDay(year, month + 1);

        return new Date(year, month, Math.min(day, endDay));
    },

    getDaysGap: function (date1, date2) {
        /// <summary>(日付 2) - (日付 1) を日数で取得する</summary>
        /// <param name="date1" type="Date">日付 1</param>
        /// <param name="date2" type="Date">日付 2</param>
        var d1 = new Date(date1.getFullYear(), date1.getMonth(), date1.getDate(), 12);
        var d2 = new Date(date2.getFullYear(), date2.getMonth(), date2.getDate(), 12);
        return Math.round((d2.getTime() - d1.getTime()) / 86400000);
    },

    getMonthEndDay: function (year, month) {
        /// <summary>末日を取得する</summary>
        /// <param name="year" type="Number">年</param>
        /// <param name="month" type="Number">月</param>
        /// <returns type="Number" />

        //日付を0にすると前月の末日を指定したことになります
        //指定月の翌月の0日を取得して末日を求めます
        //そのため、ここでは month - 1 は行いません
        var dt = new Date(year, month, 0);
        return dt.getDate();
    },

    _parseFromComponents: function (components) {
        /// <summary>日付部品群から日付を取得する</summary>
        /// <param name="components" type="Array">年から順に並んだ配列</param>
        /// <returns type="Date" />
        var year = neo.isSet(components[0], { 'blank': false }) ? Number(components[0]) : 1970;
        var month = neo.isSet(components[1], { 'blank': false }) ? Number(components[1]) : 1;
        var day = neo.isSet(components[2], { 'blank': false }) ? Number(components[2]) : 1;
        var hour = neo.isSet(components[3], { 'blank': false }) ? Number(components[3]) : 0;
        var minute = neo.isSet(components[4], { 'blank': false }) ? Number(components[4]) : 0;
        var second = neo.isSet(components[5], { 'blank': false }) ? Number(components[5]) : 0;
        var date = new Date(year, month - 1, day, hour, minute, second, 0);
        return isNaN(date.getTime()) ? null : date;
    },

    /// <summary>解析処理群</summary>
    _parsers: [
        function (dateString) {
            /// <summary>yyyyMM[dd[HH[mm[ss]]]]形式の日付を解析する(解析できない場合はnull)</summary>
            /// <param name="dateString" type="String">日付文字列</param>
            /// <returns type="Date" />
            var start = 0, end = 4, comps = [1970, 1, 1, 0, 0, 0];
            for (var i = 0; i < 6 && start < dateString.length; i++) {
                var comp = Number(dateString.substring(start, end));
                if (isNaN(comp)) {
                    return null;
                }
                comps[i] = comp;
                start = end;
                end += 2;
            }

            if (start >= 6) {
                return new Date(comps[0], comps[1] - 1, comps[2], comps[3], comps[4], comps[5], 0);
            }
            return null;
        },

        function (dateString) {
            /// <summary>yyyy-MM-dd形式の日付を解析する(解析できない場合はnull)</summary>
            /// <param name="dateString" type="String">日付文字列</param>
            /// <returns type="Date" />
            var matches = dateString.match(/^(\d{4})-(\d{2})-(\d{2})$/);
            if (!!matches && matches.length >= 4) {
                return this._parseFromComponents(matches.slice(1));
            }
            return null;
        }
    ],

    /// <summary>日付整形処理</summary>
    _formatter: {
        symbolChar: "", // 記号の種類
        symbolLength: 0, // 記号の長さ
        leteral: "", // 文字列

        beginSymbol: function (newSymbol) {
            /// <summary>記号開始部の解析：トークンを返す</summary>
            /// <param name="newSymbol" type="String">新しい記号</param>
            /// <returns type="String" />
            var value = this.getCurrentToken();
            this.symbolChar = newSymbol;
            this.symbolLength = 1;
            return value;
        },

        beginLiteral: function () {
            /// <summary>文字列開始部の解析：トークンを返す</summary>
            /// <returns type="String" />
            var value = this.getCurrentToken();
            return value;
        },

        endLiteral: function () {
            /// <summary>文字列終了部の解析：トークンを返す</summary>
            /// <returns type="String" />
            var value = this.leteral.length > 0 ? this.leteral : "'";
            this.leteral = "";
            return value;
        },

        getCurrentToken: function () {
            /// <summary>参照中のトークン文字列を取得する</summary>
            /// <returns type="String" />
            var c = this.symbolChar;
            var l = this.symbolLength;
            this.symbolChar = "";
            this.symbolLength = 0;
            return this.isSymbol(c) ? this[c](l) : "";
        },

        isSymbol: function (symbol) {
            /// <summary>記号文字かどうかを取得する</summary>
            /// <param name="symbol" type="String">文字</param>
            /// <returns type="Boolean" />
            return typeof (symbol) == "string" && symbol.length == 1 &&
                    $.isFunction(this[symbol]);
        },

        y: function (length) {
            /// <summary>y(年)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            if (length <= 2) {
                return this._zeroPad(this.date.getYear() % 100, length);
            } else {
                return this._zeroPad(this.date.getFullYear(), length);
            }
        },

        M: function (length) {
            /// <summary>M(月)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var month = this.date.getMonth() + 1;
            if (length <= 2) {
                return this._zeroPad(month, length);
            } else if (length == 3) {
                return neo.Resource.dateTime["shortMonthes"][month - 1];
            } else {
                return neo.Resource.dateTime["longMonthes"][month - 1];
            }
        },

        d: function (length) {
            /// <summary>d(日)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var day = this.date.getDate();
            return this._zeroPad(day, length);
        },

        E: function (length) {
            /// <summary>E(週の名前)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var week = this.date.getDay();
            if (length <= 3) {
                return neo.Resource.dateTime["shortWeeks"][week];
            } else {
                return neo.Resource.dateTime["longWeeks"][week];
            }
        },

        a: function (length) {
            /// <summary>a(午前/午後)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var period = (this.date.getHours() < 12) ? 0 : 1;
            return neo.Resource.dateTime["periods"][period];
        },

        h: function (length) {
            /// <summary>h(12時間)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var hour = ((this.date.getHours() + 11) % 12) + 1;
            return this._zeroPad(hour, length);
        },

        H: function (length) {
            /// <summary>H(24時間)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var hour = this.date.getHours();
            return this._zeroPad(hour, length);
        },

        K: function (length) {
            /// <summary>K(0-11)に対するhourを取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var hour = ((this.date.getHours() + 11) % 12);
            return this._zeroPad(hour, length);
        },

        k: function (length) {
            /// <summary>k(1-24)に対するhourを取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var hour = this.date.getHours() + 1;
            return this._zeroPad(hour, length);
        },

        m: function (length) {
            /// <summary>m(分)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var minute = this.date.getMinutes();
            return this._zeroPad(minute, length);
        },

        s: function (length) {
            /// <summary>s(秒)に対する値を取得する</summary>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var second = this.date.getSeconds();
            return this._zeroPad(second, length);
        },

        _zeroPad: function (value, length) {
            /// <summary>数値を0埋めする</summary>
            /// <param name="value" type="Number">値</param>
            /// <param name="length" type="Number">長さ</param>
            /// <returns type="String" />
            var str = value.toString();
            while (str.length < length) {
                str = "0" + str;
            }
            return str;
        }
    }
};
